using System;
using System.Data;
using System.Collections;

using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/4/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///		VBECS representation of the VistA user. Provides a capability to query 
	///		VistA system via VistALink to retrieve a list of active VistA users and 
	///		their divisional access information. 
	///	</summary>
	/// <remarks>
	///		IMPORTANT: VistA divisions that are not defined in VBECS will be 
	///		filtered out from user divisional access lists. 
	///	</remarks>

	#endregion

	public class VbecsVistAUser
	{
		/// <summary>
		/// VistA user name.
		/// </summary>
		protected readonly string _userName;

		/// <summary>
		/// VistA user DUZ (see property). 
		/// </summary>
		protected readonly string _duz;

		/// <summary>
		/// VistA user initials.
		/// </summary>
		protected readonly string _initials;

		/// <summary>
		/// Hashtable of VistA divisions (represented with <see cref="DivisionDefinition"/> objects) available to user. 
		/// </summary>
		protected readonly Hashtable _availableDivisions;

		/// <summary>
		/// Current list of active VBECS divisions hashed by code. 
		/// Used to filter out VistA divisions that not present in VBECS.
		/// </summary>
		private static Hashtable _vbecsDivisions;

		/// <summary>
		/// Primary constructor. Creates an instance of the class. 
		/// </summary>
		/// <param name="duz">VistA user DUZ.</param>
		/// <param name="userName">VistA user name.</param>
		/// <param name="initials">VistA user initials.</param>
		protected VbecsVistAUser( string duz, string userName, string initials )
		{
			_duz = duz;
			_userName = userName;
			_initials = initials;
			_availableDivisions = new Hashtable();
		}


		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6831"> 
		///		<ExpectedInput>VistALink is connected.</ExpectedInput>
		///		<ExpectedOutput>Non-null list of VistA users.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6832"> 
		///		<ExpectedInput>VistALink is connected.</ExpectedInput>
		///		<ExpectedOutput>Non-empty list of VistA users.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6833"> 
		///		<ExpectedInput>VistALink disconnected.</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Gets a list of all active VistA users hashed by DUZ. 
		/// Inactive user records are not returned from VistA. 
		/// </summary>
		/// <returns>List of active VistA users hashed by DUZ.</returns>
		public static Hashtable GetAllVistAUsersHashedByDuz()
		{
			DataSet _usersData = BloodBankUsers.GetBloodBankUsers();
			DataTable _userTable = _usersData.Tables[ DatabaseConstants.VistALinkTableNames.BloodBankUser ]; 

			Hashtable _userHashtable = new Hashtable( _userTable.Rows.Count );			
			
			LoadActiveVbecsDivisions(); // it's needed because a list of active divisions may change between calls. 

			foreach( DataRow _dr in _userTable.Rows )
			{
				VbecsVistAUser _user = new VbecsVistAUser( 
					StDbNullConvert.ToString( _dr[ DatabaseConstants.VistALinkColumnNames.VistaUserDuz ] ),
					StDbNullConvert.ToString( _dr[ DatabaseConstants.VistALinkColumnNames.VistaUserName ] ),
					StDbNullConvert.ToString( _dr[ DatabaseConstants.VistALinkColumnNames.VistaUserInitials ] ) );

				_user.PopulateAvailableDivisionsList( _dr );
				
				if( _user._availableDivisions.Keys.Count > 0 )
					_userHashtable.Add( _user.Duz, _user );
			}

			return _userHashtable;
		}

		/// <summary>
		/// Populates list of divisions available for the VistA user in VBECS. Filters out
		/// VistA divisions that are not defined in VBECS.
		/// </summary>
		/// <param name="vistAuserDataRow"><see cref="DataRow"/> containing VistA user information.</param>
		protected void PopulateAvailableDivisionsList( DataRow vistAuserDataRow )
		{
			DataRow[] _userDivisionRelationRows = vistAuserDataRow.GetChildRows( vistAuserDataRow.Table.ChildRelations[0] );
				
			if( _userDivisionRelationRows.Length < 1 )
				return;

			DataRow _userDivisionsRelationRow = _userDivisionRelationRows[0];				

			foreach( DataRow _divDr in _userDivisionsRelationRow.GetChildRows( _userDivisionsRelationRow.Table.ChildRelations[0] ) )
			{
				DivisionDefinition _userDivision = (DivisionDefinition)_vbecsDivisions[ _divDr[ DatabaseConstants.VistALinkColumnNames.DivisionCode ] ];

				if( _userDivision != null )
					_availableDivisions.Add( _userDivision.DivisionCode, _userDivision );
			}
		}

		/// <summary>
		/// Loads a list of active VBECS divisions. 
		/// </summary>
		/// <returns>List of active VBECS divisions.</returns>
		protected static void LoadActiveVbecsDivisions()
		{
			IList _vbecsDivisionsList = DivisionDefinition.GetActiveVbecsDivisionsDefList();
			_vbecsDivisions = new Hashtable( _vbecsDivisionsList.Count );

			foreach( DivisionDefinition _div in _vbecsDivisionsList )
				_vbecsDivisions.Add( _div.DivisionCode, _div );			
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6834"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty DUZ.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6835"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA user DUZ (VistA internal unique user ID). 
		/// </summary>
		public string Duz 
		{
			get
			{
				return _duz;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6836"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-null non-empty user name.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6837"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA user name. 
		/// </summary>
		public string UserName
		{
			get
			{
				return _userName;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6838"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>No exception is thrown.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6839"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA user initials. 
		/// </summary>
		public string Initials
		{
			get
			{
				return _initials;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6840"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-null, not-empty list of user divisions.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6841"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA divisions available for user (divisions are represented with <see cref="DivisionDefinition"/> objects) 
		/// and hashed by division code. 
		/// </summary>
		public Hashtable AvailableDivisionsHashedByCode
		{
			get
			{
				return _availableDivisions;
			}
		}
	}
}